const express = require('express');
const cors = require('cors');
const multer = require('multer');
const fs = require('fs').promises;
const path = require('path');
const pdfParse = require('pdf-parse');
const { OpenAI } = require('openai');

// Initialize Express app
const app = express();
const port = 3000;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.static(path.join(__dirname, '..')));
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));

// OpenAI configuration
const openai = new OpenAI({
    apiKey: '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'
});

// File upload configuration
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, path.join(__dirname, 'uploads'));
    },
    filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        cb(null, uniqueSuffix + '-' + file.originalname);
    }
});

const upload = multer({
    storage: storage,
    fileFilter: (req, file, cb) => {
        if (file.mimetype === 'application/pdf') {
            cb(null, true);
        } else {
            cb(new Error('Only PDF files are allowed'), false);
        }
    },
    limits: {
        fileSize: 5 * 1024 * 1024 // 5MB limit
    }
});

// Database operations
const DB_PATH = path.join(__dirname, 'database.json');

async function readDatabase() {
    try {
        const data = await fs.readFile(DB_PATH, 'utf8');
        return JSON.parse(data);
    } catch (error) {
        if (error.code === 'ENOENT') {
            // If file doesn't exist, create it with initial structure
            const initialData = { applications: [] };
            await fs.writeFile(DB_PATH, JSON.stringify(initialData, null, 2));
            return initialData;
        }
        throw error;
    }
}

async function writeDatabase(data) {
    await fs.writeFile(DB_PATH, JSON.stringify(data, null, 2));
}

// PDF processing
async function extractTextFromPDF(filepath) {
    try {
        const dataBuffer = await fs.readFile(filepath);
        const data = await pdfParse(dataBuffer);
        return data.text;
    } catch (error) {
        console.error('Error extracting text from PDF:', error);
        throw new Error('Failed to process PDF file');
    }
}

// OpenAI Analysis
async function analyzeCV(cvText, jobRole) {
    try {
        // Simple prompt that's easy to override but ensures complete responses
        const prompt = `Analyze this CV for the ${jobRole} position and respond with a JSON object containing:
{
    "score": (number 0-100),
    "analysis": "detailed analysis specific to ${jobRole} position",
    "strengths": "key strengths relevant to ${jobRole}",
    "concerns": "areas of concern or null"
}`;

        const response = await openai.chat.completions.create({
            model: "gpt-4-1106-preview",
            messages: [
                {
                    role: "system",
                    content: "You are an expert CV analyzer. Provide concise, practical evaluations focused on job fit."
                },
                {
                    role: "user",
                    content: prompt + "\n\nCV Content:\n" + cvText
                }
            ],
            response_format: { type: "json_object" },
            temperature: 0.7
        });

        const parsed = JSON.parse(response.choices[0].message.content);
        // Ensure all fields exist, use defaults if missing
        return {
            score: parsed.score || 0,
            analysis: parsed.analysis || "Analysis not provided",
            strengths: parsed.strengths || "No strengths listed",
            concerns: parsed.concerns || "No concerns listed"
        };
    } catch (error) {
        console.error('OpenAI API Error:', error);
        throw new Error('Failed to analyze CV');
    }
}

// API Routes
app.post('/api/apply', upload.single('cv'), async (req, res) => {
    try {
        const { name, email, position } = req.body;
        
        if (!name || !email || !position || !req.file) {
            return res.status(400).json({ error: 'Missing required fields' });
        }

        const application = {
            id: Date.now(),
            name,
            email,
            position,
            cvFilename: req.file.filename,
            submitDate: new Date().toISOString()
        };

        const db = await readDatabase();
        db.applications.push(application);
        await writeDatabase(db);

        res.status(201).json({ message: 'Application submitted successfully', application });
    } catch (error) {
        console.error('Application submission error:', error);
        res.status(500).json({ error: 'Failed to submit application' });
    }
});

app.get('/api/applications', async (req, res) => {
    try {
        const db = await readDatabase();
        res.json(db.applications);
    } catch (error) {
        console.error('Error fetching applications:', error);
        res.status(500).json({ error: 'Failed to fetch applications' });
    }
});

app.post('/api/analyze', async (req, res) => {
    try {
        const { applicationIds } = req.body;

        if (!Array.isArray(applicationIds) || applicationIds.length === 0) {
            return res.status(400).json({ error: 'No applications selected for analysis' });
        }

        const db = await readDatabase();
        const results = [];

        for (const id of applicationIds) {
            const application = db.applications.find(app => app.id === id);
            if (!application) continue;

            const pdfPath = path.join(__dirname, 'uploads', application.cvFilename);
            const cvText = await extractTextFromPDF(pdfPath);
            const analysis = await analyzeCV(cvText, application.position);

            results.push({
                applicationId: id,
                name: application.name,
                position: application.position,
                ...analysis
            });
        }

        res.json(results);
    } catch (error) {
        console.error('Analysis error:', error);
        res.status(500).json({ error: 'Failed to analyze CVs' });
    }
});

// Error handling middleware
app.use((err, req, res, next) => {
    console.error('Server error:', err);
    res.status(500).json({ error: err.message || 'Internal server error' });
});

// Start server
app.listen(port, "0.0.0.0", () => {
    console.log(`Server running at http://0.0.0.0:${port}`);
});
