document.addEventListener('DOMContentLoaded', () => {
    const applicationsList = document.getElementById('applications-list');
    const analyzeBtn = document.getElementById('analyze-btn');
    const refreshBtn = document.getElementById('refresh-btn');
    const positionFilter = document.getElementById('position-filter');
    const loadingOverlay = document.getElementById('loading');

    let applications = [];

    function showLoading(show) {
        loadingOverlay.style.display = show ? 'flex' : 'none';
    }

    function formatDate(dateString) {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    async function fetchApplications() {
        try {
            const response = await fetch('/api/applications');
            applications = await response.json();
            renderApplications();
        } catch (error) {
            console.error('Error fetching applications:', error);
            alert('Failed to fetch applications');
        }
    }

    function renderApplications() {
        const selectedPosition = positionFilter.value;
        const filteredApplications = selectedPosition
            ? applications.filter(app => app.position === selectedPosition)
            : applications;

        applicationsList.innerHTML = filteredApplications
            .sort((a, b) => new Date(b.submitDate) - new Date(a.submitDate))
            .map(app => `
                <div class="application-item" id="app-${app.id}">
                    <div class="application-details">
                        <input type="checkbox" data-id="${app.id}">
                        <strong>${app.name}</strong> - ${app.position}
                        <br>
                        <small>${app.email} | Submitted: ${formatDate(app.submitDate)}</small>
                        <div class="analysis-result" id="analysis-${app.id}"></div>
                    </div>
                </div>
            `).join('');
    }

    function updateAnalysisResult(applicationId, result) {
        const analysisDiv = document.getElementById(`analysis-${applicationId}`);
        if (analysisDiv) {
            analysisDiv.innerHTML = `
                <div class="score">Score: ${result.score}/100</div>
                <p><strong>Analysis:</strong> ${result.analysis}</p>
                ${result.strengths ? `<p><strong>Key Strengths:</strong> ${result.strengths}</p>` : ''}
                ${result.concerns ? `<p><strong>Areas of Concern:</strong> ${result.concerns}</p>` : ''}
            `;
        }
    }

    async function analyzeSelectedCVs() {
        const selectedBoxes = document.querySelectorAll('input[type="checkbox"]:checked');
        const selectedIds = Array.from(selectedBoxes).map(box => parseInt(box.dataset.id));

        if (selectedIds.length === 0) {
            alert('Please select at least one CV to analyze');
            return;
        }

        showLoading(true);
        try {
            const response = await fetch('/api/analyze', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ applicationIds: selectedIds })
            });

            const results = await response.json();
            
            if (Array.isArray(results)) {
                results.forEach(result => {
                    updateAnalysisResult(result.applicationId, result);
                });
            } else {
                throw new Error('Invalid response format');
            }
        } catch (error) {
            console.error('Analysis error:', error);
            alert('Failed to analyze CVs');
        } finally {
            showLoading(false);
        }
    }

    // Event Listeners
    refreshBtn.addEventListener('click', fetchApplications);
    positionFilter.addEventListener('change', renderApplications);
    analyzeBtn.addEventListener('click', analyzeSelectedCVs);

    // Initial load
    fetchApplications();
});